# Motion Capture Library

[Documentation (only reachable within network of DFKI RIC)](http://bob.dfki.uni-bremen.de/apis/dfki-interaction/experimental/mocap/)

Load, preprocess, and display trajectory data from motion capture systems or
robots.

The data might come from:
* a marker-based motion capture system such as [Qualisys](https://www.qualisys.com/)
* an IMU-based motion capture system such as [XSens](https://www.xsens.com/)
* an exoskeleton that is used for teleoperation of a robot
* a robot during kinesthetic teaching

We have data loaders for the Qualisys and XSens format. For robot and
exoskeleton data from [rock](https://www.rock-robotics.org)-based systems
we recommend to use
[pocolog2msgpack](https://github.com/rock-core/tools-pocolog2msgpack),
which combines particularly well with this library.

## Install

Clone repository and run

    python[3] setup.py install

or install a symlink with pip if you want to develop the library

    pip[3] install -e .

For some features you might need the latest version of pytransform3d. Install
it from source or with

    pip install git+https://github.com/rock-learning/pytransform3d.git@develop

## Example

```python
import mocap
from mocap.plot import plot_segmented_streams_in_rows
import matplotlib.pyplot as plt


record = mocap.load(
    metadata="test/data/myxsensmetadata.json",
    features=["position", "orientation"])
picks = record.get_segments_as_dataframes(
    label="pick", streams=["RightHand_position .*", "RightHand_orientation .*"])
picks = mocap.start_at_time_0(picks, inplace=True)
median_duration = mocap.durations_median(picks)
picks = mocap.scale_duration(picks, median_duration, dt=1.0 / 60.0)
picks = mocap.start_at_origin(picks, "RightHand_position")

plt.figure()
plot_segmented_streams_in_rows(
    picks, ["RightHand_position .*", "RightHand_orientation .*"])
plt.show()
```

## Documentation

We have a more detailed documentation sphinx documentation. Run
(with a bash or sh, not zsh)

```bash
pip install -e .[doc]
```

in the main folder of the repository to install the dependencies that you need
to build the documentation and then run, for example,

```bash
cd doc
make html
firefox build/html/index.html
```

to build the documentation and open it in firefox.

## Tests

You can run unit tests with

```bash
nosetests mocap test
```

Unit tests are in the subdirectory `mocap/test` and integration tests are
located at `test/`.

## Merge Policy / Contribution

You can contribute to the repository by making merge requests. Only
maintainers of the software is allowed to merge those. A contribution should
include unit test(s) and should be added to the sphinx documentation.
