import numpy as np
import open3d as o3d
from mocap.mano import HandState
from open3d.visualization import gui


class Figure:
    def __init__(self, window_name, width, height, ax_s=1.0):
        self.window_name = window_name
        self.width = width
        self.height = height

        gui.Application.instance.initialize()
        self.window = gui.Application.instance.create_window(
            title=self.window_name, width=self.width, height=self.height)

        em = self.window.theme.font_size
        self.layout = gui.Vert(0, gui.Margins(0.5 * em, 0.5 * em, 0.5 * em, 0.5 * em))

        self.scene_widget = gui.SceneWidget()
        self.scene_widget.scene = o3d.visualization.rendering.Open3DScene(self.window.renderer)
        self.bounds = o3d.geometry.AxisAlignedBoundingBox(-ax_s * np.ones(3), ax_s * np.ones(3))
        self.scene_widget.setup_camera(60, self.bounds, self.bounds.get_center())

        self.window.set_on_layout(self._on_layout)
        self.window.add_child(self.scene_widget)
        self.window.add_child(self.layout)

        self.menu = gui.Menu()
        QUIT_ID = 1
        self.menu.add_item("Quit", QUIT_ID)
        self.main_menu = gui.Menu()
        self.main_menu.add_menu("Menu", self.menu)
        gui.Application.instance.menubar = self.main_menu
        self.window.set_on_menu_item_activated(QUIT_ID, gui.Application.instance.quit)
        self.main_scene = self.scene_widget.scene
        self.main_scene.scene.set_sun_light([0.707, 0.0, -.707], [1.0, 1.0, 1.0], 75000)
        self.main_scene.scene.enable_sun_light(True)
        self.geometry_names = []

    def _on_layout(self, layout_context):
        # The on_layout callback should set the frame (position + size) of every
        # child correctly. After the callback is done the window will layout
        # the grandchildren.
        r = self.window.content_rect
        self.scene_widget.frame = r
        width = 30 * layout_context.theme.font_size
        height = min(
            r.height,
            self.layout.calc_preferred_size(
                layout_context, gui.Widget.Constraints()).height)
        self.layout.frame = gui.Rect(r.get_right() - width, r.y, width, height)

    def show(self):
        gui.Application.instance.run()

    def add_geometry(self, geometry, material=None):
        """Add geometry to visualizer.

        Parameters
        ----------
        geometry : Geometry
            Open3D geometry.

        material : Material, optional (default: None)
            Open3D material.
        """
        name = str(len(self.geometry_names))
        self.geometry_names.append(name)
        if material is None:
            material = o3d.visualization.rendering.Material()
        self.main_scene.add_geometry(name, geometry, material)


def make_widgets(fig, hand_state):
    em = fig.window.theme.font_size

    fig.layout.add_child(gui.Label("Options"))
    fig.layout.add_fixed(em)
    fig.layout.add_child(gui.Label("Pose Parameters"))
    for pose_parameter_idx in range(hand_state.n_pose_parameters):
        joint_limits = (-np.pi, np.pi)

        joint_control_layout = gui.Horiz()
        joint_control_layout.add_child(gui.Label(str(pose_parameter_idx + 1)))
        slider = gui.Slider(gui.Slider.DOUBLE)
        slider.set_limits(*joint_limits)
        slider.set_on_value_changed(OnSlider(fig, hand_state, pose_parameter_idx))
        joint_control_layout.add_child(slider)

        fig.layout.add_child(joint_control_layout)


class OnSlider:
    def __init__(self, fig, hand_state, idx):
        self.fig = fig
        self.hand_state = hand_state
        self.idx = idx

    def __call__(self, value):
        for g in self.fig.geometry_names:
            self.fig.main_scene.remove_geometry(g)
        self.hand_state.set_pose_parameter(self.idx, value)
        self.fig.add_geometry(self.hand_state.hand_mesh, self.hand_state.material)


left = True
fig = Figure("Left" if left else "Right", 1920, 1080, ax_s=0.2)
hand_state = HandState(left)
fig.add_geometry(hand_state.hand_mesh, hand_state.material)
make_widgets(fig, hand_state)

fig.show()
