"""
=======================================
Plot Demonstrated Peg-in-a-Hole Motions
=======================================

Multiple trajectories have been demonstrated to insert a peg in a hole.
For each demonstration, small changed to the initial position were made.
"""
from pytransform3d import visualizer as pv
import os
import sys
import glob
import numpy as np
import pandas as pd
from tqdm import tqdm


def load_pandas_dataset(path, verbose=0):
    """Load complete dataset of Cartesian trajectories from one folder."""
    if verbose >= 2:
        print("Looking for files in '%s'" % path)
    filenames = _get_all_h5(path)
    if verbose >= 1:
        print("Found %d files... " % len(filenames), end="")
        sys.stdout.flush()
    trajectories = [pd.read_hdf(filename) for filename in filenames]
    if verbose >= 1:
        print("Loading done.")
    return trajectories


def _get_all_h5(path):
    return list(sorted(glob.glob(os.path.join(path, "*.h5"))))


def plot_cart_trajectories(dataset):
    fig = pv.figure()
    for trajectory in tqdm(dataset):
        P = np.array(trajectory[["ee %s" % d for d in "XYZ"]
                                + ["ee Q%s" % d for d in "WXYZ"]])
        fig.plot_trajectory(P, n_frames=5, s=0.001)

    fig.show()


if __name__ == "__main__":
    dataset = load_pandas_dataset("test/data/mantos_kinesthetic_teaching_ur5", verbose=2)
    print("Plotting trajectories...")
    plot_cart_trajectories(dataset)
