"""
==========================
Visualize Exeskeleton Data
==========================

In this example we load a dual arm trajectory recorded from an exoskeleton
and animate it.
"""
print(__doc__)

import numpy as np
import pandas as pd
import pytransform3d.visualizer as pv
import pytransform3d.transformations as pt
from mocap.pandas_utils import match_columns, rename_stream_groups
from mocap.conversion import array_from_dataframe


trajectory = pd.read_csv("test/data/myexoskeletondata.csv", sep=" ")

patterns = ["time\.microseconds",
            "rh5_left_arm_posture_ctrl\.current_feedback\.pose\.position\.data.*",
            "rh5_left_arm_posture_ctrl\.current_feedback\.pose\.orientation\.re.*",
            "rh5_left_arm_posture_ctrl\.current_feedback\.pose\.orientation\.im.*",
            "rh5_right_arm_posture_ctrl\.current_feedback\.pose\.position\.data.*",
            "rh5_right_arm_posture_ctrl\.current_feedback\.pose\.orientation\.re.*",
            "rh5_right_arm_posture_ctrl\.current_feedback\.pose\.orientation\.im.*"]
trajectory = trajectory[match_columns(trajectory, patterns)]

group_rename = {
    "(time\.microseconds)": "Time",
    "(rh5_left_arm_posture_ctrl\.current_feedback\.pose\.position\.data).*": "left_pose",
    "(rh5_left_arm_posture_ctrl\.current_feedback\.pose\.orientation).*": "left_pose",
    "(rh5_right_arm_posture_ctrl\.current_feedback\.pose\.position\.data).*": "right_pose",
    "(rh5_right_arm_posture_ctrl\.current_feedback\.pose\.orientation).*": "right_pose"
}
trajectory = rename_stream_groups(trajectory, group_rename)

trajectory["Time"] = trajectory["Time"] / 1e6
trajectory["Time"] -= trajectory["Time"].iloc[0]
T = trajectory["Time"].to_numpy()
average_dt = np.mean(np.diff(T))
P = array_from_dataframe(
    trajectory,
    ["left_pose[0]", "left_pose[1]", "left_pose[2]",
     "left_pose.re", "left_pose.im[0]", "left_pose.im[1]", "left_pose.im[2]",
     "right_pose[0]", "right_pose[1]", "right_pose[2]",
     "right_pose.re", "right_pose.im[0]", "right_pose.im[1]", "right_pose.im[2]"])


def animation_callback(step, P, left_hand, right_hand):
    left_p, right_p = np.array_split(P[step], 2)
    left_hand.set_data(pt.transform_from_pq(left_p))
    right_hand.set_data(pt.transform_from_pq(right_p))
    return left_hand, right_hand


fig = pv.figure()
fig.plot_transform(s=0.1)
fig.plot_trajectory(P[:, :7], s=0.01)
fig.plot_trajectory(P[:, 7:], s=0.01)
left_hand = fig.plot_transform(pt.transform_from_pq(P[0, :7]), s=0.05)
right_hand = fig.plot_transform(pt.transform_from_pq(P[0, 7:]), s=0.05)
fig.view_init()
fig.animate(animation_callback, len(P), loop=True, fargs=(P, left_hand, right_hand))
fig.show()
