import os
import pickle
import hashlib


def compute_or_load(file_prefix, fun, *args, **kwargs):
    """Either execute a given function or load its result from a file.

    The function will be called if the file where we store the result does not
    exist. In this case the result will be stored in the file. Otherwise we
    will load the result from the file.

    Parameters
    ----------
    file_prefix : str
        Prefix of the file name where we either store the result or load it
        from. We will attach an (almost) unique object identifier to the
        prefix.

    fun : callable
        Function that computes the result

    *args : list
        Function arguments

    **kwargs : dict
        Function keyword arguments

    Returns
    -------
    res : anything
        Result of the function call
    """
    try:
        filename = _unique_filename(file_prefix, *args, **kwargs)
        with open(filename, "rb") as f:
            res = pickle.load(f)
        print("[mocap] Loading cached data from %s" % filename)
        return res
    except:
        res = fun(*args, **kwargs)
        save_result(file_prefix, res, *args, **kwargs)
        return res


def save_result(file_prefix, res, *args, **kwargs):
    filename = _unique_filename(file_prefix, *args, **kwargs)
    dirname = os.path.dirname(filename)
    if dirname and not os.path.exists(dirname):
        os.makedirs(dirname)
    with open(filename, "wb") as f:
        pickle.dump(res, f)


def _unique_filename(file_prefix, *args, **kwargs):
    return "%s_%s.pickle" % (file_prefix, identifier(*args, **kwargs))


def identifier(*args, **kwargs):
    """Generate unique object identifier based on object values.

    Parameters
    ----------
    args : list
        Arguments

    kwargs : dict
        Keyword arguments

    Returns
    -------
    identifier : string of length 16
        (Approximately) unique object identifier that is independent of the program state
    """
    h = hashlib.md5()
    for arg in args:
        h.update(_to_hashable_type(arg))
    for k, v in kwargs.items():
        if k == "verbose":
            continue
        h.update(k.encode("utf-8"))
        h.update(_to_hashable_type(v))
    return h.hexdigest()


def _to_hashable_type(obj):
    try:
        memoryview(obj)
        return obj
    except TypeError:
        return str(obj).encode("utf-8")
