try:
    import vtk
    import numpy as np
    import re
    from pytransform3d.transformations import transform_from_pq
    from pytransform3d.rotations import matrix_from_euler_xyz
    from .conversion import array_from_dataframe


    def plot3d(dataframes, groups, alpha=1.0, linewidth=1.0, window_size=(500, 500),
               azimuth=180, elevation=120, zoom=1.0, axis=True):
        """TODO document"""
        trajectories = []
        group_res = [re.compile(g) for g in groups]
        for df in dataframes:
            for group_re in group_res:
                columns = sorted([c for c in df.columns if group_re.match(c)])
                X = array_from_dataframe(df, columns)
                trajectories.append(X)
                # TODO colorize

        window = VisualizerWindow(
            window_size=window_size, azimuth=azimuth, elevation=elevation,
            zoom=zoom)

        if axis:
            frame = Frame(np.eye(4), label=None, s=1.0)
            window.draw(frame)

        for X in trajectories:
            trajectory = PositionTrajectory(
                X, color=[0.0, 0.0, 0.0], opacity=alpha, linewidth=linewidth)
            window.draw(trajectory)
        window.show()
        return window


    class VisualizerWindow:
        def __init__(self, window_size=(500, 500), azimuth=180, elevation=120,
                     zoom=1.0):
            self.azimuth = azimuth
            self.elevation = elevation
            self.zoom = zoom

            self.colors = vtk.vtkNamedColors()
            self.colors.SetColor("BkgColor", [255, 255, 255, 255])
            self.renderer = vtk.vtkRenderer()
            self.render_window_ = vtk.vtkRenderWindow()
            self.render_window_.SetWindowName("Trajectory")
            self.render_window_.SetSize(*window_size)
            self.render_window_.AddRenderer(self.renderer)
            self.renderer.SetViewport(0, 0, *window_size)

        def draw(self, obj):
            obj.draw(self.renderer)

        def show(self):
            self.renderer.SetBackground(self.colors.GetColor3d("BkgColor"))
            self.renderer.ResetCamera()
            self.renderer.GetActiveCamera().Azimuth(self.azimuth)
            self.renderer.GetActiveCamera().Elevation(self.elevation)
            self.renderer.GetActiveCamera().Zoom(self.zoom)
            self.renderer.ResetCameraClippingRange()
            self.render_window_.Render()


    class Frame:
        def __init__(self, A2B=np.eye(4), label=None, s=1.0, lw=1.0, opacity=1.0):
            self.draw_label = label is not None
            self.label = label

            self.s = s

            colors = vtk.vtkNamedColors()
            self.x_axis = vtk.vtkLineSource()
            self.x_axis_mapper = vtk.vtkPolyDataMapper()
            self.x_axis_mapper.SetInputConnection(self.x_axis.GetOutputPort())
            self.x_axis_actor = vtk.vtkActor()
            self.x_axis_actor.SetMapper(self.x_axis_mapper)
            self.x_axis_actor.GetProperty().SetColor(colors.GetColor3d("Red"))
            self.x_axis_actor.GetProperty().SetLineWidth(lw)

            self.y_axis = vtk.vtkLineSource()
            self.y_axis_mapper = vtk.vtkPolyDataMapper()
            self.y_axis_mapper.SetInputConnection(self.y_axis.GetOutputPort())
            self.y_axis_actor = vtk.vtkActor()
            self.y_axis_actor.SetMapper(self.y_axis_mapper)
            self.y_axis_actor.GetProperty().SetColor(colors.GetColor3d("Green"))
            self.y_axis_actor.GetProperty().SetLineWidth(lw)

            self.z_axis = vtk.vtkLineSource()
            self.z_axis_mapper = vtk.vtkPolyDataMapper()
            self.z_axis_mapper.SetInputConnection(self.z_axis.GetOutputPort())
            self.z_axis_actor = vtk.vtkActor()
            self.z_axis_actor.SetMapper(self.z_axis_mapper)
            self.z_axis_actor.GetProperty().SetColor(colors.GetColor3d("Blue"))
            self.z_axis_actor.GetProperty().SetLineWidth(lw)

            if self.draw_label:
                self.text_property = vtk.vtkTextProperty()
                self.text_property.SetFontSize(16)
                self.text_property.SetColor(0.0, 0.0, 0.0)

                self.text_actor = vtk.vtkTextActor3D()
                self.text_actor.SetInput(self.label)
                self.text_actor.SetTextProperty(self.text_property)
                self.text_actor.SetPosition(0, 0, 0)

            if opacity != 1.0:
                self.set_opacity(opacity)

            self.set_data(A2B)

        def get_opacity(self):
            return self.z_axis_actor.GetProperty().GetOpacity()

        def set_opacity(self, opacity):
            self.x_axis_actor.GetProperty().SetOpacity(opacity)
            self.y_axis_actor.GetProperty().SetOpacity(opacity)
            self.z_axis_actor.GetProperty().SetOpacity(opacity)

        def draw(self, renderer):
            renderer.AddActor(self.x_axis_actor)
            renderer.AddActor(self.y_axis_actor)
            renderer.AddActor(self.z_axis_actor)
            if self.draw_label:
                renderer.AddActor(self.text_actor)

        def set_data(self, A2B):
            if A2B.ndim == 1:
                A2B = transform_from_pq(A2B)

            R = A2B[:3, :3]
            p = A2B[:3, 3]

            for d, b in enumerate([self.x_axis, self.y_axis, self.z_axis]):
                b.SetPoint1(p)
                b.SetPoint2(p + self.s * R[:, d])


    class PositionTrajectory:
        def __init__(self, X, color=(0.0, 0.0, 0.0), opacity=1.0, linewidth=1.0):
            self.polyline = vtk.vtkPolyLineSource()
            self.polyline_mapper = vtk.vtkPolyDataMapper()
            self.polyline_mapper.SetInputConnection(self.polyline.GetOutputPort())
            self.polyline_actor = vtk.vtkActor()
            self.polyline_actor.GetProperty().SetColor(*color)
            self.polyline_actor.GetProperty().SetOpacity(opacity)
            self.polyline_actor.GetProperty().SetLineWidth(linewidth)
            self.polyline_actor.SetMapper(self.polyline_mapper)

            self.set_data(X)

        def get_linewidth(self):
            return self.polyline_actor.GetProperty().GetLineWidth()

        def set_linewidth(self, linewidth):
            self.polyline_actor.GetProperty().SetLineWidth(linewidth)

        def set_color(self, color):
            self.polyline_actor.GetProperty().SetColor(*color)

        def get_opacity(self):
            return self.polyline_actor.GetProperty().GetOpacity()

        def set_opacity(self, opacity):
            self.polyline_actor.GetProperty().SetOpacity(opacity)

        def draw(self, renderer):
            renderer.AddActor(self.polyline_actor)

        def set_data(self, X):
            self.points = vtk.vtkPoints()
            self.points.Allocate(len(X))
            for i in range(len(X)):
                self.points.InsertNextPoint(X[i, :3].tolist())

            self.polyline.SetPoints(self.points)


    class PoseTrajectory:
        def __init__(self, X, color=(0.0, 0.0, 0.0), opacity=1.0, linewidth=1.0,
                     s=1.0, n_frames = 10):
            self.position_trajectory = PositionTrajectory(
                X, color, opacity, linewidth)
            self.key_frames = [Frame(np.eye(4), s=s, lw=linewidth)
                               for _ in range(n_frames)]
            self.set_data(X)
            # TODO arrow indicating direction (optional)

        def get_linewidth(self):
            return self.position_trajectory.get_linewidth()

        def set_linewidth(self, linewidth):
            self.position_trajectory.set_linewidth(linewidth)

        def set_color(self, color):
            self.position_trajectory.set_color(color)

        def get_opacity(self):
            return self.position_trajectory.get_opacity()

        def set_opacity(self, opacity):
            self.position_trajectory.set_opacity(opacity)
            for key_frame in self.key_frames:
                key_frame.set_opacity(opacity)

        def deactivate_orientation(self):
            for key_frame in self.key_frames:
                key_frame.set_opacity(0.0)

        def draw(self, renderer):
            self.position_trajectory.draw(renderer)
            for key_frame in self.key_frames:
                key_frame.draw(renderer)

        def set_data(self, X):
            self.position_trajectory.set_data(X)
            key_frames_indices = np.linspace(
                0, len(X) - 1, len(self.key_frames), dtype=np.int)
            for i, key_frame_idx in enumerate(key_frames_indices):
                self.key_frames[i].set_data(X[key_frame_idx])


    class Obj:
        def __init__(self, filename, scale=None, rotate=None):
            self.scale = scale

            self.reader = vtk.vtkOBJReader()
            self.reader.SetFileName(filename)
            self.reader.Update()

            self.mapper = vtk.vtkPolyDataMapper()

            self.transform = vtk.vtkTransform()
            self.R = vtk.vtkMatrix4x4()
            self.R.Identity()

            self.P = vtk.vtkMatrix4x4()
            self.P.Identity()

            if rotate is not None:
                Ra = matrix_from_euler_xyz(rotate).T
                for i in range(3):
                    for j in range(3):
                        self.R.SetElement(i, j, Ra[i, j])

            self.transformPolyData = vtk.vtkTransformPolyDataFilter()
            self.transformPolyData.SetTransform(self.transform)
            self.transformPolyData.SetInputConnection(
                self.reader.GetOutputPort())
            self.mapper.SetInputConnection(
                self.transformPolyData.GetOutputPort())

            self.actor = vtk.vtkActor()
            self.actor.SetMapper(self.mapper)

            self.set_data(np.eye(4))

        def draw(self, renderer):
            renderer.AddActor(self.actor)

        def set_data(self, A2B):
            if A2B.ndim == 1:
                A2B = transform_from_pq(A2B)

            R = vtk.vtkMatrix4x4()
            R.Identity()
            for i in range(3):
                for j in range(3):
                    R.SetElement(i, j, A2B[i, j])

            position = A2B[:3, 3]
            for i in range(3):
                self.P.SetElement(i, 3, position[i])

            self.transform.SetMatrix(self.P)
            self.transform.Concatenate(self.R)
            self.transform.Concatenate(R)
            if self.scale is not None:
                self.transform.Scale(self.scale, self.scale, self.scale)


    class Sphere:
        def __init__(self, position=(0, 0, 0), radius=1.0):
            self.sphere = vtk.vtkSphereSource()
            self.mapper = vtk.vtkPolyDataMapper()
            self.mapper.SetInputConnection(self.sphere.GetOutputPort())
            self.actor = vtk.vtkActor()
            self.actor.SetMapper(self.mapper)

            self.set_data(position, radius)

        def draw(self, renderer):
            renderer.AddActor(self.actor)

        def set_data(self, position, radius=None):
            self.sphere.SetCenter(*position[:3])
            if radius is not None:
                self.sphere.SetRadius(radius)


    class TrajectoryAnimationCallback:
        def __init__(self, actors, trajectory):
            self.actors = actors
            self.trajectory = trajectory
            self.timer_id = None

        def execute(self, obj, event):
            for t in range(len(self.trajectory)):
                for a in self.actors:
                    a.set_data(self.trajectory[t])
                obj.GetRenderWindow().Render()
            obj.DestroyTimer(self.timer_id)

except ImportError:
    import warnings
    warnings.warn("Visualization not available, requires VTK")
