"""Data normalization."""
from scipy import interpolate
import pandas as pd
import numpy as np
from .pandas_utils import get_all_markers


# TODO unify interfaces: should all functions handle trajectories or just one?


def to_frequency(trajectory, frequency, kind="linear"):
    """Resample trajectory with new frequency.

    Parameters
    ----------
    trajectory : DataFrame
        Time series data

    frequency : float
        Target frequency

    kind : str, optional (default: 'linear')
        Kind of interpolation. See scipy.interpolate.interp1d for details.

    Returns
    -------
    trajectory : DataFrame
        Resampled trajectory
    """
    if "Time" not in trajectory:
        raise ValueError("A pandas dataframe needs the column 'Time' to "
                         "change the frequency. Only got: %s"
                         % (", ".join(trajectory.columns)))
    dt = 1.0 / frequency
    start_time = trajectory["Time"].iloc[0]
    end_time = trajectory["Time"].iloc[-1]
    new_time = np.arange(start_time, end_time + dt, dt)

    interpolated_trajectory = {"Time": new_time}
    columns = list(trajectory.columns)
    columns.remove("Time")
    for col in columns:
        f = interpolate.interp1d(
            trajectory["Time"], trajectory[col], kind=kind,
            fill_value="extrapolate", assume_sorted=True)
        interpolated_trajectory[col] = f(new_time)
    return pd.DataFrame(interpolated_trajectory)


def start_at_time_0(trajectories, inplace=False):
    """Manipulate time stream to start at 0.

    Parameters
    ----------
    trajectories : list of DataFrame
        A collection of time series data

    inplace : bool, optional (default: False)
        Change data inplace without making a copy

    Returns
    -------
    trajectories : list of DataFrame
        A collection of time series data with the column 'Time' starting at 0
    """
    if inplace:
        result = trajectories
    else:
        result = []
    for trajectory in trajectories:
        if "Time" not in trajectory:
            raise ValueError("All trajectories must have column 'Time'.")

        if inplace:
            t = trajectory
        else:
            t = trajectory.copy()
        t["Time"] -= t["Time"].iloc[0]
        if not inplace:
            result.append(t)
    return result


def scale_duration(trajectory, target_duration, dt, kind="linear"):
    """Scale trajectory to target duration.

    Parameters
    ----------
    trajectory : DataFrame
        Time series data

    target_duration : float
        Target duration

    dt : float
        Time between each step

    kind : str, optional (default: 'linear')
        Kind of interpolation. See scipy.interpolate.interp1d for details.

    Returns
    -------
    trajectory : DataFrame
        Temporally scaled time series data
    """
    if "Time" not in trajectory:
        raise ValueError("Trajectory must have column 'Time'.")

    duration = trajectory["Time"].iloc[-1] - trajectory["Time"].iloc[0]
    time_scaling = target_duration / duration + 10 * np.finfo(np.float).eps
    t = np.array(trajectory["Time"]) * time_scaling
    t -= t[0]
    new_time = np.arange(0, target_duration + dt, dt)

    interpolated_trajectories = {"Time": new_time}
    columns = list(trajectory.columns)
    columns.remove("Time")
    for col in columns:
        f = interpolate.interp1d(
            t, trajectory[col], kind=kind, fill_value="extrapolate",
            assume_sorted=True)
        interpolated_trajectories[col] = f(new_time)
    return pd.DataFrame(interpolated_trajectories)


def start_at_origin(trajectory, marker):
    """Normalize trajectory of marker to start at position (0, 0, 0).

    Parameters
    ----------
    trajectory : DataFrame
        Time series data

    marker : str
        Marker name, each markers must have three corresponding streams:
        marker + ' X', marker + ' Y', and marker + ' Z'.

    Returns
    -------
    new_trajectory : DataFrame
        Time series data with marker starting at position (0, 0, 0)
    """
    marker_cols = ["%s %s" % (marker, axis) for axis in "XYZ"]
    start_pos = [trajectory[m].iloc[0] for m in marker_cols]
    new_trajectory = trajectory.copy()
    for current_marker in get_all_markers(trajectory):
        for d, axis in enumerate("XYZ"):
            new_trajectory["%s %s" % (current_marker, axis)] -= start_pos[d]
    return new_trajectory
