"""Compute orientation."""
import numpy as np
from pytransform3d import rotations as pr
import warnings


def orientation_from_three_positions(X1, X2, X3):
    """Estimate orientation trajectory from marker triangle.

    This diagram is currently only visible correctly in the code (TODO)::

           X1
          /  \
         /    \
        X2__   \
            \__X3

    The x-axis of the generated orientation will point along the perpendicular
    from the line X2-X3 to X1. The y-axis corresponds to direction vector from
    X3 to X2. The z-axis points upwards to form an orthonormal basis.

    Parameters
    ----------
    X1 : array-like, shape (n_steps, 3)
        First marker trajectory (see diagram)

    X2 : array-like, shape (n_steps, 3)
        Second marker trajectory (see diagram)

    X3 : array-like, shape (n_steps, 3)
        Third marker trajectory (see diagram)

    Returns
    -------
    Q : array, shape (n_steps, 4)
        Orientation trajectory (rotation quaternions)
    """
    n_steps = X1.shape[0]
    Q = np.empty((n_steps, 4))

    for t in range(n_steps):
        X3_to_X2 = X2[t] - X3[t]
        X3_to_X1 = X1[t] - X3[t]
        # Orthogonal projection: orth_proj_of_X3_to_X1_on_X3_to_X2 starts at
        # X3 and ends at the point that is the start point of the vector that
        # is orthogonal to X3_to_X2 and ends at the end point of X3_to_X1.
        orth_proj_of_X3_to_X1_on_X3_to_X2 = (
                X3_to_X1.dot(X3_to_X2) / X3_to_X2.dot(X3_to_X2) * X3_to_X2)
        foot_of_perpendicular = X3[t] + orth_proj_of_X3_to_X1_on_X3_to_X2

        x_axis = pr.norm_vector(X1[t] - foot_of_perpendicular)
        y_axis = pr.norm_vector(X2[t] - X3[t])
        z_axis = pr.perpendicular_to_vectors(x_axis, y_axis)
        R = np.column_stack([x_axis, y_axis, z_axis])
        q = pr.quaternion_from_matrix(R)

        if not np.all(np.isfinite(q)):
            warnings.warn("Could not compute valid rotation at step %d" % t)
            Q[t] = np.array([1.0, 0.0, 0.0, 0.0])
        else:
            # we have to ensure smoothness of quaternion components because
            # of ambiguity: q = -q
            if t > 1:
                if (np.linalg.norm(q - Q[t - 1]) >
                        np.linalg.norm(-q - Q[t - 1])):
                    q = -q
            Q[t] = q

    return Q
