"""
This module parses exported mvnx data files from the XSens sensor suit.

Author: Mariela De Lucas Alvarez <mariela.de_lucas_alvarez@dfki.de>
        Alexander Fabisch <alexander.fabisch@dfki.de>
"""
from xml.etree import ElementTree
import pandas as pd
import numpy as np
import warnings


NS = {'default': 'http://www.xsens.com/mvn/mvnx',
      'xsi': 'http://www.w3.org/2001/XMLSchema-instance',
      'schemaLocation':
          'http://www.xsens.com/mvn/mvnx '
          'http://www.xsens.com/mvn/mvnx/schema.xsd'}
SEGMENT_FEATURES = ("position",
                    "orientation",
                    "velocity",
                    "angularVelocity",
                    "acceleration",
                    "angularAcceleration")
SENSOR_FEATURES = ("sensorFreeAcceleration",
                   "sensorMagneticField",
                   "sensorOrientation")


def read_xsens_mvnx(filename, section="segments",
                    features=SEGMENT_FEATURES, verbose=0):
    """Reads exported mvnx data files from the XSens sensor suit.

    Parameters
    ----------
    filename : str
        Source file

    section : str, optional (default: 'segments')
        'segments' or 'sensors'

    features : list, optional (default: all segment features)
        Data streams that should be extracted. Possible options for
        section == 'segments': position, orientation, velocity,
        angularVelocity, acceleration, angularAcceleration. Possible options
        for section == 'sensor': sensorFreeAcceleration, sensorMagneticField,
        sensorOrientation.

    verbose : int, optional (default: 0)
        Verbosity level

    Returns
    -------
    df : DataFrame
        Raw data streams from source file
    """
    if verbose >= 2:
        print("Loading XSens MVNX file '%s'" % filename)

    tree = ElementTree.parse(filename)
    root_elem = tree.getroot()
    subject_elem = root_elem.find("default:subject", NS)

    elems = subject_elem.find("default:" + section, NS)
    text_data = [elem.attrib["label"] for elem in elems]

    frames_elem = subject_elem.find("default:frames", NS)
    elems = frames_elem.findall("default:frame", NS)
    data = [_get_text(frame_elem, features)
            for frame_elem in elems if frame_elem.attrib["type"] == "normal"]

    df = _make_dataframe(text_data, features, data)

    return df


def _make_dataframe(headers, features, data):
    # Scalar first convention is used for quaternions.
    # See https://base.xsens.com/hc/en-us/articles/115004491045-Orientation-output-specifications
    quat_suffix = [' QW', ' QX', ' QY', ' QZ']
    vec_suffix = [' X', ' Y', ' Z']

    header_names = [h + '_' + f for f in features for h in headers]
    column_names = ['Index', 'Time']

    for col in header_names:
        is_quaternion = 'orientation' in col or 'Orientation' in col
        if is_quaternion:
            column_names.extend(
                [col + quat_suffix[0], col + quat_suffix[1],
                 col + quat_suffix[2], col + quat_suffix[3]])
        else:
            column_names.extend(
                [col + vec_suffix[0], col + vec_suffix[1],
                 col + vec_suffix[2]])

    df = pd.DataFrame(data, columns=column_names)
    df.set_index('Index', inplace=True)

    # Convert time to seconds
    df["Time"] = pd.to_numeric(df["Time"])
    df["Time"] /= 1000.0

    return df


def _get_text(frame_elem, features):
    row = [frame_elem.attrib["index"], frame_elem.attrib["time"]]
    for feature in features:
        curr = frame_elem.find("default:" + feature, NS)
        if feature is not None:
            row.extend(np.fromstring(curr.text, dtype=float, sep=' '))
        else:
            warnings.warn("'%s' is not member of a frame" % feature)
            continue
    return row
