import numpy as np
import pandas as pd
from mocap.pandas_utils import match_columns, rename_stream_groups
from mocap.conversion import array_from_dataframe
from nose.tools import assert_equal, assert_in, assert_almost_equal
from numpy.testing import assert_array_equal


def test_use_case_load_exo_data():
    trajectory = pd.read_csv("test/data/myexoskeletondata.csv", sep=" ")

    assert_equal(len(trajectory), 201)
    assert_equal(len(trajectory.columns), 181)

    patterns = ["time\.microseconds",
                "rh5_left_arm_posture_ctrl\.current_feedback\.pose\.position\.data.*",
                "rh5_left_arm_posture_ctrl\.current_feedback\.pose\.orientation\.re.*",
                "rh5_left_arm_posture_ctrl\.current_feedback\.pose\.orientation\.im.*",
                "rh5_right_arm_posture_ctrl\.current_feedback\.pose\.position\.data.*",
                "rh5_right_arm_posture_ctrl\.current_feedback\.pose\.orientation\.re.*",
                "rh5_right_arm_posture_ctrl\.current_feedback\.pose\.orientation\.im.*"]
    trajectory = trajectory[match_columns(trajectory, patterns)]

    assert_equal(len(trajectory), 201)
    assert_equal(len(trajectory.columns), 15)

    group_rename = {
        "(time\.microseconds)": "Time",
        "(rh5_left_arm_posture_ctrl\.current_feedback\.pose\.position\.data).*": "left_pose",
        "(rh5_left_arm_posture_ctrl\.current_feedback\.pose\.orientation).*": "left_pose",
        "(rh5_right_arm_posture_ctrl\.current_feedback\.pose\.position\.data).*": "right_pose",
        "(rh5_right_arm_posture_ctrl\.current_feedback\.pose\.orientation).*": "right_pose"
    }
    trajectory = rename_stream_groups(trajectory, group_rename)

    assert_equal(len(trajectory), 201)
    assert_equal(len(trajectory.columns), 15)
    assert_in("left_pose[0]", trajectory)
    assert_in("right_pose.im[2]", trajectory)

    trajectory["Time"] = trajectory["Time"] / 1e6
    trajectory["Time"] -= trajectory["Time"].iloc[0]
    T = trajectory["Time"].to_numpy()
    assert_almost_equal(T[0], 0.0)
    average_dt = np.mean(np.diff(T))
    assert_almost_equal(average_dt, 0.005, delta=0.0003)
    P = array_from_dataframe(
        trajectory,
        ["left_pose[0]", "left_pose[1]", "left_pose[2]",
         "left_pose.re", "left_pose.im[0]", "left_pose.im[1]", "left_pose.im[2]",
         "right_pose[0]", "right_pose[1]", "right_pose[2]",
         "right_pose.re", "right_pose.im[0]", "right_pose.im[1]", "right_pose.im[2]"])
    assert_array_equal(P.shape, (201, 14))