import mocap
from mocap.plot import plot_segmented_streams_in_rows
import matplotlib.pyplot as plt
from nose.tools import assert_equal, assert_almost_equal, assert_in


def test_load_from_metadata():
    record = mocap.load(metadata="test/data/mymetadata.json")

    throws = record.get_segments_as_dataframes(
        label="throw", streams=["Hand .*", "Back .*"])
    assert_equal(len(throws), 3)
    assert_in("Hand top X", throws[0])
    assert_in("Hand top Y", throws[0])
    assert_in("Hand top Z", throws[0])
    assert_in("Back top X", throws[0])
    assert_in("Back top Y", throws[0])
    assert_in("Back top Z", throws[0])

    throws = mocap.to_frequency(throws, 15)
    assert_almost_equal(
        throws[0]["Time"].iloc[1] - throws[0]["Time"].iloc[0], 1.0 / 15.0)

    if __name__ == "__main__":
        plt.figure()
        record.plot_streams_in_rows(["Hand .*", "Back .*"])
        plt.figure()
        plot_segmented_streams_in_rows(throws, ["Hand .*", "Back .*"])
        plt.show()


def test_load_from_xsens():
    record = mocap.load(
        xsens_filename="test/data/myxsens.mvnx", features=["position"])
    streams = record.get_available_streams()
    assert_in("RightHand_position X", streams)
    assert_in("RightHand_position Y", streams)
    assert_in("RightHand_position Z", streams)

    if __name__ == "__main__":
        plt.figure()
        record.plot(["RightHand_position .*"])
        plt.show()


def test_load_from_xsens_metadata():
    record = mocap.load(
        metadata="test/data/myxsensmetadata.json",
        features=["position", "orientation"])

    all_streams = record.get_available_streams()
    assert_in("RightHand_position X", all_streams)
    assert_in("RightHand_orientation QW", all_streams)

    picks = record.get_segments_as_dataframes(
        label="pick", streams=["RightHand_position .*",
                               "RightHand_orientation .*"])
    assert_equal(len(picks), 51)

    picks = mocap.start_at_time_0(picks, inplace=True)
    assert_equal(picks[-1]["Time"].iloc[0], 0.0)

    median_duration = mocap.durations_median(picks)
    assert_equal(median_duration, 1.1599999999999966)

    picks = mocap.scale_duration(picks, median_duration, dt=1.0 / 60.0)
    assert_equal(len(picks[0]), 71)

    picks = mocap.start_at_origin(picks, "RightHand_position")
    assert_equal(picks[0]["RightHand_position X"].iloc[0], 0.0)

    columns = (["RightHand_position %s" % d for d in "XYZ"]
               + ["RightHand_orientation Q%s" % d for d in "WXYZ"])
    demonstrations = [mocap.array_from_dataframe(p, columns) for p in picks]
    assert_equal(len(demonstrations), 51)
    assert_equal(demonstrations[0].shape[1], 7)

    if __name__ == "__main__":
        plt.figure()
        plot_segmented_streams_in_rows(
            picks, ["RightHand_position .*", "RightHand_orientation .*"])
        plt.show()

        from mocap.deprecated_visualization import plot3d
        window = plot3d(
            picks, groups=["RightHand_position .*"],
            azimuth=0, elevation=-45, window_size=(1024, 768), axis=False)
        input("Press Enter...")


if __name__ == "__main__":
    test_load_from_xsens_metadata()
